// const OTTOREPLY_API_URL = 'http://localhost:5001/api/elead'; // dev
const OTTOREPLY_API_URL = 'https://ottoreply.com/api/elead';

// Polling interval when job is active (30 seconds)
const POLL_INTERVAL_MS = 30000;

// DOM elements
const states = {
  loadingStatus: document.getElementById('loading-status'),
  notElead: document.getElementById('not-elead'),
  notLinked: document.getElementById('not-linked'),
  ready: document.getElementById('ready'),
  jobProgress: document.getElementById('job-progress'),
  syncComplete: document.getElementById('sync-complete'),
  cooldown: document.getElementById('cooldown'),
  error: document.getElementById('error')
};

// Ready state elements
const orgNameSpan = document.getElementById('org-name');
const lastSyncTimeSpan = document.getElementById('last-sync-time');
const lastSyncStatsP = document.getElementById('last-sync-stats');
const syncBtn = document.getElementById('sync-btn');

// Progress state elements
const progressFill = document.getElementById('progress-fill');
const progressText = document.getElementById('progress-text');
const etaText = document.getElementById('eta-text');
const hotCount = document.getElementById('hot-count');
const warmCount = document.getElementById('warm-count');

// Complete state elements
const resultDeals = document.getElementById('result-deals');
const resultHot = document.getElementById('result-hot');
const resultWarm = document.getElementById('result-warm');
const ottoLink = document.getElementById('otto-link');
const doneBtn = document.getElementById('done-btn');

// Cooldown state elements
const cooldownCountdown = document.getElementById('cooldown-countdown');
const cooldownOttoBtn = document.getElementById('cooldown-otto-btn');

// Error state elements
const errorMessage = document.getElementById('error-message');
const retryBtn = document.getElementById('retry-btn');

// Footer elements
const ottoFooter = document.getElementById('otto-footer');
const openOttoBtn = document.getElementById('open-otto-btn');

// Logo header
const logoHeader = document.getElementById('logo-header');

// Not-elead state buttons
const goToEleadBtn = document.getElementById('go-to-elead-btn');
const goToOttoBtn = document.getElementById('go-to-otto-btn');

// State
let currentOrg = null;
let pollInterval = null;
let cooldownInterval = null;
let cooldownEndTime = null;

// Base URL for OttoReply (matches API URL)
const OTTOREPLY_BASE_URL = OTTOREPLY_API_URL.replace('/api/elead', '');

// Show only the specified state
function showState(stateName) {
  Object.values(states).forEach(el => el.classList.add('hidden'));
  states[stateName]?.classList.remove('hidden');

  // Hide logo when showing not-elead state (it has its own image)
  if (stateName === 'notElead') {
    logoHeader.classList.add('hidden');
  } else {
    logoHeader.classList.remove('hidden');
  }
}

// Get all cookies for eleadcrm.com
async function getEleadCookies() {
  return new Promise((resolve) => {
    chrome.cookies.getAll({ domain: 'eleadcrm.com' }, (cookies) => {
      resolve(cookies.map(c => ({ name: c.name, value: c.value })));
    });
  });
}

// Check if user is on eleadcrm.com (active tab)
async function isOnElead() {
  const tabs = await chrome.tabs.query({ active: true, currentWindow: true });
  if (tabs.length === 0) return false;
  const currentTab = tabs[0];
  return currentTab.url && currentTab.url.includes('eleadcrm.com');
}

// Fetch status from OttoReply
async function fetchStatus(cookies) {
  const response = await fetch(`${OTTOREPLY_API_URL}/status`, {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({ cookies })
  });
  return response.json();
}

// Start sync job
async function startSync(cookies) {
  const response = await fetch(`${OTTOREPLY_API_URL}/sync`, {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({ cookies })
  });
  return response.json();
}

// Format relative time
function formatRelativeTime(isoString) {
  if (!isoString) return 'Never';

  // Server returns UTC times without Z suffix - ensure proper parsing
  let dateStr = isoString;
  if (!dateStr.endsWith('Z') && !dateStr.includes('+') && !dateStr.includes('-', 10)) {
    dateStr += 'Z';
  }
  const date = new Date(dateStr);
  const now = new Date();
  const diffMs = now - date;
  const diffMins = Math.floor(diffMs / 60000);
  const diffHours = Math.floor(diffMs / 3600000);
  const diffDays = Math.floor(diffMs / 86400000);

  if (diffMins < 1) return 'Just now';
  if (diffMins < 60) return `${diffMins} minute${diffMins > 1 ? 's' : ''} ago`;
  if (diffHours < 24) return `${diffHours} hour${diffHours > 1 ? 's' : ''} ago`;
  return `${diffDays} day${diffDays > 1 ? 's' : ''} ago`;
}

// Update UI with organization info
function updateOrgInfo(org) {
  currentOrg = org;
  orgNameSpan.textContent = org.name;
  // Show the footer with OttoReply link
  ottoFooter.classList.remove('hidden');
}

// Update UI with last sync info
function updateLastSync(lastSync) {
  if (!lastSync) {
    lastSyncTimeSpan.textContent = 'Never';
    lastSyncStatsP.textContent = '';
    return;
  }

  lastSyncTimeSpan.textContent = formatRelativeTime(lastSync.completed_at);
  lastSyncStatsP.textContent = `${lastSync.deals_scraped} deals, ${lastSync.hot_leads_found} hot, ${lastSync.warm_leads_found} warm`;
}

// Format ETA seconds into human-readable string
function formatETA(seconds) {
  if (seconds < 60) {
    return `~${seconds}s remaining`;
  } else if (seconds < 3600) {
    const mins = Math.floor(seconds / 60);
    const secs = seconds % 60;
    return `~${mins}m ${secs}s remaining`;
  } else {
    const hours = Math.floor(seconds / 3600);
    const mins = Math.floor((seconds % 3600) / 60);
    return `~${hours}h ${mins}m remaining`;
  }
}

// Update UI with job progress
function showJobProgress(job) {
  showState('jobProgress');

  const total = job.total_deals || 1;
  const scraped = job.deals_scraped || 0;
  const percent = Math.round((scraped / total) * 100);

  progressFill.style.width = `${percent}%`;

  if (job.status === 'pending') {
    progressText.textContent = 'Starting sync...';
  } else {
    progressText.textContent = `Processing: ${scraped}/${total} deals`;
  }

  // Display ETA if available (only shown after 20+ items processed)
  if (job.eta_seconds !== undefined && job.eta_seconds !== null) {
    etaText.textContent = formatETA(job.eta_seconds);
    etaText.style.display = 'block';
  } else {
    etaText.style.display = 'none';
  }

  hotCount.textContent = job.hot_leads_found || 0;
  warmCount.textContent = job.warm_leads_found || 0;
}

// Show sync complete state
function showSyncComplete(lastSync) {
  showState('syncComplete');

  resultDeals.textContent = lastSync.deals_scraped || 0;
  resultHot.textContent = lastSync.hot_leads_found || 0;
  resultWarm.textContent = lastSync.warm_leads_found || 0;

  // Set link to OttoReply dashboard
  if (currentOrg) {
    ottoLink.href = `${OTTOREPLY_BASE_URL}/${currentOrg.slug}/deal-assessments`;
  }
}

// Show cooldown state with countdown
function showCooldownState(cooldownData) {
  showState('cooldown');
  stopCooldownTimer();

  if (cooldownData.next_sync_allowed_at) {
    cooldownEndTime = new Date(cooldownData.next_sync_allowed_at);
    updateCooldownCountdown();
    cooldownInterval = setInterval(updateCooldownCountdown, 1000);
  }
}

// Update cooldown countdown display
function updateCooldownCountdown() {
  if (!cooldownEndTime) return;

  const now = new Date();
  const diff = cooldownEndTime - now;

  if (diff <= 0) {
    // Cooldown expired - re-initialize
    stopCooldownTimer();
    init();
    return;
  }

  const hours = Math.floor(diff / 3600000);
  const mins = Math.floor((diff % 3600000) / 60000);
  const secs = Math.floor((diff % 60000) / 1000);

  if (cooldownCountdown) {
    if (hours > 0) {
      cooldownCountdown.textContent = `${hours}h ${mins}m ${secs}s`;
    } else if (mins > 0) {
      cooldownCountdown.textContent = `${mins}m ${secs}s`;
    } else {
      cooldownCountdown.textContent = `${secs}s`;
    }
  }
}

// Stop cooldown timer
function stopCooldownTimer() {
  if (cooldownInterval) {
    clearInterval(cooldownInterval);
    cooldownInterval = null;
  }
  cooldownEndTime = null;
}

// Start polling for job status
function startPolling(cookies) {
  if (pollInterval) return; // Already polling

  pollInterval = setInterval(async () => {
    try {
      const status = await fetchStatus(cookies);

      if (status.success) {
        if (status.active_job) {
          showJobProgress(status.active_job);
        } else {
          // Job completed
          stopPolling();
          showSyncComplete(status.last_sync);
        }
      }
    } catch (e) {
      console.error('Polling error:', e);
    }
  }, POLL_INTERVAL_MS);
}

// Stop polling
function stopPolling() {
  if (pollInterval) {
    clearInterval(pollInterval);
    pollInterval = null;
  }
}

// Handle sync button click
async function handleSyncClick() {
  try {
    const cookies = await getEleadCookies();
    const result = await startSync(cookies);

    if (result.success) {
      showJobProgress(result.job);
      startPolling(cookies);
    } else if (result.error === 'sync_cooldown') {
      // Handle cooldown restriction
      showCooldownState(result);
    } else {
      errorMessage.textContent = result.error || 'Failed to start sync';
      showState('error');
    }
  } catch (e) {
    console.error('Sync error:', e);
    errorMessage.textContent = 'Failed to connect to OttoReply';
    showState('error');
  }
}

// Handle done button click
function handleDoneClick() {
  init(); // Reset to initial state
}

// Initialize popup
async function init() {
  showState('loadingStatus');
  stopPolling();

  try {
    // Check if on elead
    const onElead = await isOnElead();
    if (!onElead) {
      showState('notElead');
      return;
    }

    // Get cookies and fetch status
    const cookies = await getEleadCookies();
    const status = await fetchStatus(cookies);

    if (!status.success) {
      if (status.error && status.error.includes('No organization')) {
        showState('notLinked');
      } else {
        errorMessage.textContent = status.error || 'Failed to get status';
        showState('error');
      }
      return;
    }

    // Update org info
    updateOrgInfo(status.organization);
    updateLastSync(status.last_sync);

    // Check for active job first
    if (status.active_job) {
      showJobProgress(status.active_job);
      startPolling(cookies);
    } else if (status.cooldown && status.cooldown.in_cooldown) {
      // Show cooldown state if restriction is active
      showCooldownState(status.cooldown);
    } else {
      showState('ready');
    }

  } catch (e) {
    console.error('Init error:', e);
    errorMessage.textContent = 'Failed to connect to OttoReply. Is the server running?';
    showState('error');
  }
}

// Handle view progress button click (opens OttoReply in new tab)
function handleViewProgressClick() {
  if (currentOrg) {
    chrome.tabs.create({ url: `${OTTOREPLY_BASE_URL}/${currentOrg.slug}/sync-jobs` });
  }
}

// Handle open OttoReply button click (footer button)
function handleOpenOttoClick() {
  if (currentOrg) {
    chrome.tabs.create({ url: `${OTTOREPLY_BASE_URL}/${currentOrg.slug}/deal-assessments` });
  } else {
    chrome.tabs.create({ url: OTTOREPLY_BASE_URL });
  }
}

// Event listeners
syncBtn.addEventListener('click', handleSyncClick);
retryBtn.addEventListener('click', init);
doneBtn.addEventListener('click', handleDoneClick);
document.getElementById('view-progress-btn').addEventListener('click', handleViewProgressClick);
openOttoBtn.addEventListener('click', handleOpenOttoClick);
if (cooldownOttoBtn) {
  cooldownOttoBtn.addEventListener('click', handleOpenOttoClick);
}

// Not-elead state buttons
goToEleadBtn.addEventListener('click', () => {
  chrome.tabs.create({ url: 'https://www.eleadcrm.com' });
});
goToOttoBtn.addEventListener('click', () => {
  chrome.tabs.create({ url: OTTOREPLY_BASE_URL });
});

// Stop polling when popup closes
window.addEventListener('unload', stopPolling);

// Initialize on load
init();
